﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Threading.Tasks;
    using Domain.Helpers;
    using Domain.Services;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Models.Patient;
    using Newtonsoft.Json;
    using Shared.DataFilters;
    using Shared.EntityModels;
    using Utilities;

    /// <summary>
    /// The patient emergency contacts.
    /// </summary>
    [Authorize]
    [Route("api/patient-insurances")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class PatientInsurancesController : BaseController
    {
        /// <summary>
        /// The patient emergency services.
        /// </summary>
        private readonly IPatientInsuranceService patientInsuranceService;

        /// <summary>
        /// The AES helper.
        /// </summary>
        private readonly IAESHelper aesHelper;

        /// <inheritdoc />
        public PatientInsurancesController(IPatientInsuranceService patientInsuranceService, IAESHelper aesHelper)
        {
            this.patientInsuranceService = patientInsuranceService;
            this.aesHelper = aesHelper;
        }

        /// <summary>
        /// The fetch emergency contacts async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("fetch")]
        [ProducesResponseType(typeof(List<PatientInsuranceModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAsync([FromBody] FindPatientRequest model)
        {
            model = (FindPatientRequest)EmptyFilter.Handler(model);
            var patientId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedPatientId));
            var patientInsurances = await this.patientInsuranceService.FetchAsync(patientId);

            return this.Success(patientInsurances);
        }

        /// <summary>
        /// The add emergency contacts async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Authorize]
        [Route("add")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> AddAsync([FromBody] PatientInsuranceModels model)
        {
            var msgJson = new List<string>();
            var message = string.Empty;
            foreach (var item in model.Insurances)
            {
                var msg = string.Empty;

                var response = await this.patientInsuranceService.AddAsync(item);

                if(response == -1)
                {
                    msg = "Already Exists";
                }
                else if(response > 0)
                {
                    msg = $@"Record Inserted Successfully";
                }
                else
                {
                    msg = $@"{response} Record not Inserted";
                }
                msgJson.Add(msg);
            }
            var s = String.Join("", msgJson.ToArray());
            return this.Success(String.Join("",msgJson.ToArray()));
        }

        /// <summary>
        /// The update emergency contacts async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPut]
        [Authorize]
        [Route("update")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> UpdateAsync([FromBody] PatientInsuranceModel model)
        {
            var response = await this.patientInsuranceService.UpdateAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given Insurance has already been exists with us.");
                case 0:
                    return this.ServerError();
                default:
                    return this.Success("Insurance details updated successfully.");
            }
        }

        /// <summary>
        /// The delete emergency contacts async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Authorize]
        [Route("delete")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> DeleteAsync([FromBody] PatientInsuranceModel model)
        {
            var response = await this.patientInsuranceService.DeleteAsync((int)model.PatientInsuranceId);
            if (response != 0)
            {
                return this.Success("Insurance details deleted successfully.");
            }

            return this.ServerError();
        }
    }
}